// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_AUDIO_SOURCE_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_AUDIO_SOURCE_H

#include "common.h"
#include "ProtocolEndpointBase.h"
#include "util/Mutex.h"
#include "IAudioSourceCallbacks.h"

/**
 * This endpoint is used to send an audio stream from the car to the phone. This is typically
 * from an on-board microphone placed near the driver. The typical initialization sequence for
 * such an endpoint is described below:
 * <br>
 * <pre>
 *      galReceiver->init();
 *      ... Initialization code ...
 *      AudioSource* endpoint = new AudioSource(serviceId, galReceiver->messageRouter());
 *      endpoint->registerCallbacks(callbacks); // Subclassed from IAudioSourceCallbacks.
 *      endpoint->setConfig(...);
 *      galReceiver->registerService(endpoint);
 *      ... Other Initialization code ...
 *      galReceiver->start();
 *
 *      microphoneRequestCallback() gets invoked.
 *          open microphone.
 *
 *      User starts speaking, voice data is available.
 *          call send() as long as less than maxUnackedFrames are outstanding.
 *
 *      ackCallback() gets invoked.
 *          reduce the value of currently outstanding frames and send a new frame if necessary.
 *
 *      microphoneRequestCallback() gets invoked.
 *          close microphone.
 * </pre>
 */
class AudioSource : public ProtocolEndpointBase {
public:
    AudioSource(uint8_t id, MessageRouter *router, int mediaCodecType) :
            ProtocolEndpointBase(id, router, false),
            mMediaCodecType(mediaCodecType) { }

    void addDiscoveryInfo(ServiceDiscoveryResponse *sdr);
    int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    void registerCallbacks(const shared_ptr<IAudioSourceCallbacks>& callbacks) {
            mCallbacks = callbacks;
    }

    /**
     * Sets up the parameters of the audio stream. This should be called during initialization.
     * @param samplingRate The sampling rate.
     * @param numberOfBits Number of bits per sample.
     * @param numberOfChannels Number of channels.
     */
    void setConfig(int samplingRate, int numberOfBits, int numberOfChannels);
    /**
     * Use this function to send pcm voice data to the phone.
     * @param timestamp The timestamp of this data in nanoseconds.
     * @param data The pcm data.
     */
    void send(uint64_t timestamp, const shared_ptr<IoBuffer>& data);
protected:
    void handleAck(const Ack& ack);
    int handleMicrophoneRequest(const MicrophoneRequest &req);

private:
    void microphoneResponse(int32_t status, int32_t sessionId);

    shared_ptr<IAudioSourceCallbacks> mCallbacks;
    int mMediaCodecType;
    int32_t mSamplingRate;
    int32_t mNumberOfBits;
    int32_t mNumberOfChannels;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_AUDIO_SOURCE_H
